/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Globalization;

using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Default;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;

namespace Borland.Eco.Persistence.Oracle
{
	public sealed class StringAsVarChar2: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return String.Format(CultureInfo.InvariantCulture, "VARCHAR2({0:d})", length); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.String));
			parameter.DbType = DbType.String;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(value, parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class Int32AsNumber: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMBER(10,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int32));
			parameter.DbType = DbType.Int32;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt32(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (int)(decimal)columnValue;
		}
	}

	public sealed class Int16AsNumber: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMBER(5,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int16));
			parameter.DbType = DbType.Int16;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt16(value), parameter);
		}

		private static System.Int16 obj2Int16(System.Object value)
		{
			if (value is System.Int16)
				return (System.Int16)value;
			else if (value is System.Int32)
				return (System.Int16)value;
			else if (value is System.Double)
				return (System.Int16)Math.Round((System.Double)value);
			else if (value is System.Decimal)
				return (System.Int16)Math.Round((System.Decimal)value);
			else
				throw new ArgumentException(PersistenceStringRes.sInt16AsNumber_CannotConvert(value.ToString()));
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			if ((columnValue == null) || (columnValue == System.DBNull.Value))
				return null;
			else
				return obj2Int16(columnValue);
		}
	}

	public sealed class DoubleAsNumber: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMBER"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Double));
			parameter.DbType = DbType.Double;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDouble(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (double)(decimal)columnValue;
		}
	}

	///<summary>
	///Mapper between an DateTime value and its persistent representation as DATE.
	///</summary>
	public sealed class DateTimeAsDate: AbstractSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "DATE"; // do not localize
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="sqlDatabaseConfig"/> is null</exception>
		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if (sqlDatabaseConfig == null) throw new ArgumentNullException("sqlDatabaseConfig"); // do not localize
			if (((modeledValue == null) || (modeledValue.Length == 0)) && (sqlDatabaseConfig.DateTimeFormat.Length > 0))
				return sqlDatabaseConfig.DefaultDateTime.ToString(sqlDatabaseConfig.DateTimeFormat);
			else
				return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.DateTime));
			parameter.DbType = DbType.DateTime;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDateTime(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.DateTime));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}

		public override bool IsEqual(System.Object value, System.Object columnValue)
		{
			return MappingUtils.DateTimeIsEqual(value, ColumnToValue(columnValue));
		}
	}

	///<summary>
	///Mapper between a DateTime value and its persistent representation as TIMESTAMP.
	///</summary>
	public sealed class DateTimeAsTimeStamp: AbstractSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "TIMESTAMP"; // do not localize
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="sqlDatabaseConfig"/> is null</exception>
		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if (sqlDatabaseConfig == null) throw new ArgumentNullException("sqlDatabaseConfig"); // do not localize
			if (((modeledValue == null) || (modeledValue.Length == 0)) && (sqlDatabaseConfig.DateTimeFormat.Length > 0))
				return sqlDatabaseConfig.DefaultDateTime.ToString(sqlDatabaseConfig.DateTimeFormat);
			else
				return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.DateTime));
			parameter.DbType = DbType.DateTime;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDateTime(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.DateTime));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}

		public override bool IsEqual(System.Object value, System.Object columnValue)
		{
			return MappingUtils.DateTimeIsEqual(value, ColumnToValue(columnValue));
		}
	}

	public sealed class TimeSpanAsDate: AbstractSingleColumnAttribute, ISingleColumnAttributemapping
	{
		private DateTime ZeroDateTime = new DateTime(1900, 1, 1, 0, 0, 0);
		public String ColumnType(int length)
		{
			return "DATE"; // do not localize
		}

		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.TimeSpan));
			parameter.DbType = DbType.DateTime;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
			{
				TimeSpan ts = (TimeSpan)value;
				parameter.Value = ZeroDateTime + ts;
			}
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToTimespan(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.DateTime));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
			{
				DateTime dt = (DateTime)columnValue;
				return dt - ZeroDateTime;
			}
		}

		public override	bool IsEqual(System.Object value, System.Object columnValue)
		{
			return MappingUtils.TimeSpanIsEqual(value, ColumnToValue(columnValue));
		}
	}

	public sealed class TimeSpanAsLongInteger: AbstractSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.TimeSpan));
			parameter.DbType = DbType.Double;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
			{
				long ticks = ((TimeSpan)value).Ticks;
				parameter.Value = ticks;
			}
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToTimespan(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
			{
				long ticks = (long)(decimal)columnValue;
				TimeSpan ts = new TimeSpan(ticks);
				return ts;
			}
		}

		public override	bool IsEqual(System.Object value, System.Object columnValue)
		{
			return MappingUtils.TimeSpanIsEqual(value, ColumnToValue(columnValue));
		}
	}

	public sealed class BooleanAsInteger: AbstractSingleColumnAttribute, ISingleColumnAttributemapping, INonBooleanBooleanMapping
	{
		public String ColumnType(int length)
		{
			return "INTEGER"; // do not localize
		}

		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if ((modeledValue == null) || (modeledValue.Length == 0))
				return "0"; // do not localize
			else
				return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Boolean));
			parameter.DbType = DbType.Int32;
			if (value == null)
				parameter.Value = DBNull.Value;
			else if ((bool)value == true)
				parameter.Value = 1;
			else
				parameter.Value = 0;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToBoolean(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (columnValue == DBNull.Value)
				return null;
			else if ((System.Decimal)columnValue == 0)
				return false;
			else
				return true;
		}

		public string TrueString
		{
			get { return "1"; } // do not localize
		}
	}

	public sealed class DecimalAsDecimal: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			// DECIMAL wants (precision, scale)
			// where precision is number of digits,
			// and scale is number of decimals.
			// Oracle wants no larger than 38 in size.
			return String.Format(CultureInfo.InvariantCulture, "DECIMAL({0:d},2)", Math.Min(length, 38)); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Decimal));
			parameter.DbType = DbType.Decimal ;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDecimal(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class SingleAsSingle: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "FLOAT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Single));
			parameter.DbType = DbType.Single;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDouble(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Single)(decimal)columnValue;
		}
	}

	public sealed class ByteAsSmallInt: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "SMALLINT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Byte));
			parameter.DbType = DbType.Int16;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = (System.Int16)(System.Byte)value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt16(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Byte)(decimal)columnValue;
		}
	}

	public sealed class Int64AsLongInteger: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null) 
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int64));
			parameter.DbType = DbType.Decimal;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt64(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Int64)(System.Decimal)columnValue;
		}
	}

	///<summary>
	///Base class mapper between an AutoInc value and its persistent representation.
	///This class must be inherited.
	///</summary>
	public abstract class AbstractAutoInc: AbstractNumericSingleColumnAttribute
	{
		///<summary>This method should never be called. Autoinc values are never stored!</summary>
		///<exception cref="NotSupportedException">Thrown always.</exception>
		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			throw new NotSupportedException(PersistenceStringRes.sDoNotCallInt32AsAutoIncValueToParameter);
		}

		///<summary>This method should never be called. Autoinc values are never stored!</summary>
		///<exception cref="NotSupportedException">Thrown always.</exception>
		public void StringToParameter(string value, IDataParameter parameter)
		{
			throw new NotSupportedException(PersistenceStringRes.sDoNotCallInt32AsAutoIncStringToParameter);
		}
		private static void ExecuteSql(IExecQuery q, string sql)
		{
			q.AssignSqlText(sql);
			q.ExecSql();
		}
		private static void SafeExecuteSql(IExecQuery q, string sql)
		{
			try
			{
				ExecuteSql(q, sql);
			}
			catch 
			{
				//Swallow exceptions of failing statements
			}
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="database"/> is null</exception>
		public void MakeColumnAutoInc(string tableName, string columnName, IDatabase database)
		{
			if (database == null) throw new ArgumentNullException("database"); // do not localize
			string generatorName = tableName + '_' + columnName; // do not localize
			generatorName = generatorName.ToUpper(CultureInfo.InvariantCulture); // interbase stores the name as uppercase...

			string DropSequenceSql = String.Format(CultureInfo.InvariantCulture, "DROP SEQUENCE {0}", generatorName); // do not localize
			string DropTriggerSql = String.Format(CultureInfo.InvariantCulture, "DROP TRIGGER {0}_BI", tableName); // do not localize
			
			string CreateSequenceSql = String.Format(CultureInfo.InvariantCulture,
			"CREATE SEQUENCE {0} INCREMENT BY 1 START WITH 1 ORDER", // do not localize
			generatorName);
			string CreateTriggerSql = String.Format(CultureInfo.InvariantCulture,
			"CREATE OR REPLACE TRIGGER {0}_BI " + // do not localize
				"BEFORE INSERT " +  // do not localize
				"	ON {0} " +  // do not localize
				"	FOR EACH ROW " + // do not localize
				"begin " + // do not localize
				"	select {1}.nextval into :new.id from dual;" + // do not localize
				"end", tableName, generatorName); // do not localize
			IExecQuery q = database.GetExecQuery();
			try
			{
				SafeExecuteSql(q, DropSequenceSql);
				SafeExecuteSql(q, DropTriggerSql);
				ExecuteSql(q, CreateSequenceSql);
				ExecuteSql(q, CreateTriggerSql);
			}
			finally
			{
				database.ReleaseExecQuery(q);
			}
		}
	}

	public sealed class Int32AsAutoInc: AbstractAutoInc, IAutoIncAttributemapping
	{
		public String ColumnType(int length)
		{
			return "INTEGER"; // do not localize
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int32));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}
}

/*
Oracle Type			BdpType			BdpSubType			System.Type

CHAR			String			stFixed			String
NCHAR			String			stFixed			String
VARCHAR			String						String
NVARCHAR			String						String
VARCHAR2			String						String
NVARCHAR2			String						String
NUMBER			Decimal						Decimal
DATE			Date						DateTime
BLOB			Blob			stHBinary			Byte[]
CLOB			Blob			stHMemo			Char[]
LONG			Blob			stMemo			Char[]
LONG RAW			Blob			stBinary			Byte[]
BFILE			Blob			stBFile			Char[]
ROWID			String						String
*/



